/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Runtime.Serialization;
using System.Collections;
using System.Collections.Specialized;

using Borland.Eco.Interfaces;
using Borland.Eco.UmlRt;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Ocl.Support;
using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Subscription;
using Borland.Eco.DataRepresentation;

namespace Borland.Eco.Services
{
	///<summary>
	///Allows you to discover whether a particular object or property in the ECO
	///space has been modified.
	///</summary>
	public interface IStateService
	{
		///<summary>
		/// Returns true if the object has any modified members or if it has been created/deleted.
		///</summary>
		bool IsDirty(IObject obj);
		///<summary>
		/// Returns true if the property has been modified since it was loaded from the database.
		///</summary>
		bool IsDirty(IProperty prop);
		///<summary>
		/// Returns true if the object has not yet been saved to the database.
		///</summary>
		bool IsNew(IObject obj);
	}

	///<summary>
	/// What it says. LinkConflict is when one end of an association is fetched and it doesn't match the other end in the cache.
	/// 'All' means that anything could have changed.
	///</summary>
	public enum ChangeKind {MemberChanged, ObjectCreated, ObjectDeleted, LinkConflict, All}
	///<summary>
	/// The kind of action that is to occur when the change is Apply()ed.
	/// <para>Ignore means that nothing will happen when the change is applied. The change is removed from the list of changes. If the reason for the change was an optimistic locking failure, then the same failure will occur the next time</para>
	/// <para>Discard means that the value in the ecospace will be discarded, a new value will be fetched from the database the next time the value of the element is read</para>
	/// <para>Keep means that the value in the ecospace will be kept, and a subsequent update will overwrite the value in the database</para>
	/// <para>Verify will check if the value of the element is really changed. If the value of the element in the Ecospace is the same as in the database, the change will be removed from the list of changes</para>
	/// <para>Reread that a new value will immediately be fetched from the database </para>
	/// <para>Undecided - a change that is undecided can not be applied. If you call Apply on this nothing will happen, and the change will still be in the list of changes</para>
	///</summary>
	public enum ChangeActionKind {Ignore, Discard, Keep, Verify, Reread, Undecided}

	///<summary>
	/// A "change" represents knowledge about discrepancies between an EcoSpace and its persistent storage.
	/// A change can be created as a result of a call to a change handler, or an optimistic locking failure. 
	/// Not all changes means that there is a conflict between changes in the EcoSpace and the database
	///</summary>
	public interface IChange
	{
		///<summary>
		/// The Kind explains what kind of change this Change element represents. 
		///</summary>
		ChangeKind Kind { get; }
		///<summary>
		/// This property controls what happens when the Apply-method is called. Default value is Undecided.
		///</summary>
		ChangeActionKind Action	{ get; set; }
		///<summary>
		/// Indicates whether this change has been applied or not.
		///</summary>
		bool Applied { get; }
		///<summary>
		/// Call this to execute the action specified by the Action property.
		///</summary>
		void Apply();
		///<summary>
		/// This is the element that has been changed in the database.
		///</summary>
		IElement ChangedElement { get; }
		///<summary>
		/// If this property is true, that means that the element refered to be ChangedElement 
		/// has been changed in both the ecospace and the database. Some kind of conflict resolution has to be applied.
		///</summary>
		bool IsDirectConflict { get; }
	}
	///<summary>
	/// This Exception is thrown by IPersistenceService.UpdateDatabaseWithList if
    /// there is a optimistic locking conflict.
	///</summary>
	[Serializable]
	public class OptimisticLockException: InvalidOperationException
	{
		public OptimisticLockException(): base() {}
		public OptimisticLockException(string message): base(message) {}
		public OptimisticLockException(string message, Exception inner): base(message, inner) {}
#if !CF
		protected OptimisticLockException(SerializationInfo info, StreamingContext context): base(info, context) {}
#endif
	}

	///<summary>
	/// This interface groups the persistence operations.
	///</summary>
	public interface IPersistenceService
	{
		///<summary>
		/// Sends a set of updates to the persistent storage (normally a database, but possibly an xml file or some other storage)
		/// the set of objects to update could either be retrieved from the IDirtyListService, or build manually.
		/// see also EnsureEnclosure
		///</summary>
		void UpdateDatabaseWithList(IObjectList list);
		///<summary>
		/// Sends a single object to the persistent storage (normally a database, but possibly an xml file or some other storage)
		/// see also EnsureEnclosure
		///</summary>
		void UpdateDatabase(IObjectInstance obj);
		///<summary>
		/// True if the persistence service is connected to a synchronization server
		///</summary>
		bool SupportsSync { get; }
		void Refresh(bool fetchValues);
		///<summary>
		/// This procedure will retrieve information of changes made by other clients from a central change server. 
		/// It will not apply any changes to the ecospace, it will just update the list that can be retrieved by calling GetChanges
		///</summary>
		void RetrieveChanges();
		///<summary>
		/// This function will return a list of known changes.  
		///</summary>
		IChange[] GetChanges();
		///<summary>
		/// Call ApplyAllChanges to apply the changes to the ecospace. For further info, see IChange
		///</summary>
		void ApplyAllChanges();
		///<summary>
		/// Returns the set of objects that matches the condition. The condition is normally generated by calling IOclService.CreateRetrieveCondition
		///</summary>
		IObjectList GetAllWithCondition(AbstractCondition condition);
		IObjectList GetAllWithCondition(AbstractCondition condition, int maxAnswers, int offset);
		///<summary>
		/// Builds the required enclosure in order to get a consistent set of objects to update. It will (recursively) include all new refered by any of the objects in the set.
		/// it will also include all objects that refer to a deleted object in the set. This will be performed internally when you call UpdateDatabase
		/// but if you want to have control over which objects will be implicitly added to the list, you can call this method first.
		///</summary>
		void EnsureEnclosure(IObjectList list);
		///<summary>
		/// Removes the contents of a list of objects from the cache.
		/// It is still OK to keep references to the objects.
		///</summary>
		void Unload(IObjectList objects);
		///<summary>
		/// Removes the contents of an object from the cache.
		/// It is still OK to keep references to the object.
		///</summary>
		void Unload(IObject obj);
		///<summary>
		/// Ensures that the contents of the objects in the range are fetched.
		/// This allows for more efficient fetching.
		///</summary>
		void EnsureRange(IObjectList objects, int fromIndex, int toIndex);
		///<summary>
		/// Ensures that the association memberName along with all referred
		/// objects are fetched for all objects in the list. This allows for more efficient fetching.
		///</summary>
		void EnsureRelatedObjects(IObjectList objects, string memberName);
		///<summary>
		/// True if EcoSpace is Persisent, i.e. if the methods on the
		/// PersistenceService may actually be called.
		///</summary>
		bool IsPersistent {get;}
	}

	/// <summary>
	/// This service keeps track of objects that have changed compared to their state in the database/xml-file
	/// modified, created, deleted.
	/// </summary>
	public interface IDirtyListService
	{
		/// <summary>
		/// Returns all modified objects.
		/// </summary>
		IObjectList AllDirtyObjects();
		///<summary>
		/// Returns true if "AllDirtyObjects.Count &lt;&gt; 0".
		///</summary>
		bool HasDirtyObjects();
		///<summary>
		/// Subscribe to if Dirty list has changed, or rather may have changed.
		///</summary>
		void Subscribe(ISubscriber subscriber);
	}

	///<summary>
	///Allows you to query the eco space for all instances of a class.
	///</summary>
	public interface IExtentService
	{
		/// <summary>
		/// Returns the list of all objects of the specified class (including subclasses).
		/// </summary>
		IObjectList AllInstances(IClass c);
		IObjectList AllInstances(System.Type t);
		IObjectList AllInstances(string className);
		/// <summary>
		/// The same as "AllInstnaces" except it will only return objects that have already been loaded into memory.
		/// </summary>
		IObjectList AllLoadedInstances(IClass c);
		IObjectList AllLoadedInstances(System.Type t);
		IObjectList AllLoadedInstances(string className);
		/// <summary>
		/// The subscriber will be notified if a new object in the specified class is created.
		/// </summary>
		void SubscribeToObjectAdded(ISubscriber subscriber, IClass c);
		void SubscribeToObjectAdded(ISubscriber subscriber, System.Type t);
		void SubscribeToObjectAdded(ISubscriber subscriber, string className);
		/// <summary>
		/// The subscriber will be notified if an object in the specified class is deleted.
		/// </summary>
		void SubscribeToObjectRemoved(ISubscriber subscriber, IClass c);
		void SubscribeToObjectRemoved(ISubscriber subscriber, System.Type t);
		void SubscribeToObjectRemoved(ISubscriber subscriber, string className);
		/// <summary>
		/// Unload the extent. This will free the memory used for storing the
		/// extent, and will ensure that the extent is reread from persistent storage
		/// the next time AllInstances is called for the class. Note that unLoad() for a class
		/// also unloads all superclasses
		/// </summary>
		void Unload(IClass c);	
	}

	/// <summary>
	/// Provides methods for you to create new instances of the classes in your
	/// model. The IObjectFactoryService interface methods create new objects
	/// using their type information. This approach is more generic than directly
	/// creating a new object by calling the C# new method, or the Delphi
	/// Create method.
	/// </summary>
	public interface IObjectFactoryService
	{
		/// <summary>
		/// Creates an object of the given IClass.
		/// </summary>
		IObjectInstance CreateNewObject(IClass c);
		///<summary>
		/// Creates an object of the given .Net type (for example: CreateNewObject(typeof(Person));)
		///</summary>
		IObjectInstance CreateNewObject(Type t);
		///<summary>
		///Creates a new object given a name of the type: CreateNewObject("Person");
		///</summary>
		IObjectInstance CreateNewObject(string className);
	}

	///<summary>
	///Provides a programmatic interface for creating what are essentially
	///VariableHandle components. Variables created with this service can be
	///used directly with the IOclService.
	///</summary>
	public interface IVariableFactoryService
	{
		///<summary>
		///Creates a variable of the specified type.
		///</summary>
		IElement CreateVariable(IClassifier classifier);
		///<summary>
		///Creates a variable of the specified type.
		///</summary>
		IElement CreateVariable(Type type);
		///<summary>
		///Creates a variable of the specified type.
		///</summary>
		IElement CreateVariable(string typeName);
		///<summary>
		///Creates a constant variable of the specified type with the specified value.
		///</summary>
		IElement CreateConstant(IClassifier classifier, System.Object asObject);
		///<summary>
		///Creates a constant variable with the specified value.
		///</summary>
		IElement CreateConstant(System.Object asObject);
		///<summary>
		///Creates an objectlist that can contain any object from the model.
		///</summary>
		IObjectList CreateUntypedObjectList(bool allowDuplicates);
		///<summary>
		///Creates an objectlist that can contain objects from the specified class (and subclasses).
		///</summary>
		IObjectList CreateTypedObjectList(IClass umlClass, bool allowDuplicates);
		IObjectList CreateTypedObjectList(Type type, bool allowDuplicates);
		///<summary>
		///Creates a list that can contain any IElement.
		///</summary>
		IElementCollection CreateUntypedElementCollection(bool allowDuplicates);
		///<summary>
		///Returns a variable list that can be used as a parameter to call IOclService.EvaluateAndSubscribe.
		///</summary>
		IModifiableVariableList CreateVariableList();
		///<summary>
		///Returns a variable list that can be used as a parameter to call IOclService.EvaluateAndSubscribe.
		///</summary>
		IModifiableVariableList CreateVariableList(string name, object valueAsObject);
		///<summary>
		///Returns a variable list that can be used as a parameter to call IOclService.EvaluateAndSubscribe.
		///</summary>
		IModifiableVariableList CreateVariableList(string name, IElement value);
		///<summary>
		///Returns a variable list that can be used as a parameter to call IOclService.EvaluateAndSubscribe.
		///</summary>
		IModifiableVariableList CreateVariableList(string name, IClassifier classifier);
		///<summary>
		///Returns a variable list that can be used as a parameter to call IOclService.EvaluateAndSubscribe.
		///</summary>
		IModifiableVariableList CreateVariableList(string name, Type type);
		///<summary>
		///Returns an Empty list that can be used as a parameter to call IOclService.EvaluateAndSubscribe.
		///</summary>
		IExternalVariableList EmptyVariableList {get;}
		///<summary>
		///A Null object reference.
		///</summary>
		Borland.Eco.ObjectRepresentation.IObjectInstance NullReference {get;}
	}

	///<summary>
	///Allows you to get the type system of the model, and to validate the
	///model programmatically.
	///</summary>
	public interface ITypeSystemService
	{
		///<summary>
		/// returns the type system (model) used by the service provider (eco space).
		///</summary>
		IEcoTypeSystem TypeSystem { get; }
		///<summary>
		/// The services that are statically available given only the typesystem
		///</summary>
		IEcoServiceProvider StaticEcoServices { get; }
	}

	///<summary>
	///For domain classes that have been marked as versioned, this interface
	///allows you to get a specific version of an object from persistent storage.
	///</summary>
	public interface IVersionService
	{
		///<summary>
		///Gets the same element at a different version.
		///</summary>
		IElement GetVersion(int version, IElement element);
		///<summary>
		///Returns the version of the element.
		///</summary>
		int ElementVersion(IElement element);
		///<summary>
		///Converts a physical time to a version.
		///</summary>
		int VersionAtTime(DateTime time);
		///<summary>
		///Converts a version to a physical time.
		///</summary>
		DateTime TimeForVersion(int version);
		///<summary>
		///The version of current objects (an object with this version can be modified, other objects will be readonly).
		///</summary>
		int CurrentVersion { get; }
		///<summary>
		///The version number of the most recent update performed by the ecospace.
		///</summary>
		int MaxSavedVersion { get; }

		///<summary>
		///This function returns a condition that can be used with
		///GetAllWithCondition to determine at what times a certain object has
		///been modified. It is only applicable for classes that are stored in
		///a versioned database.
		///By default, only changes to the embedded members (normally attributes and
		///singlelinks) will be detected. If names of non embedded members are specified, these will also be checked. 
		///</summary>
		AbstractCondition GetChangePointCondition(IObjectList objects, int startVersion, int stopVersion, string[] members);
		AbstractCondition GetChangePointCondition(IObjectList objects, int startVersion, int stopVersion);
		AbstractCondition GetChangePointCondition(IObject obj, int startVersion, int stopVersion, string[] members);
		AbstractCondition GetChangePointCondition(IObject obj, int startVersion, int stopVersion);

	}

	///<summary>
	/// The methods of this interface performs static analysis of ocl-expressions.
	/// Mainly intended to be used in designtime to calculate static types of handles
	///</summary>
	public interface ITypeService
	{
		///<summary>
		/// these two overloads returns the classifier that the result of evaluating the <paramref name="expression"/> would have.
		///</summary>
		IClassifier ExpressionType(string expression, IClassifier context, bool reRaise);
		IClassifier ExpressionType(string expression, IClassifier context, bool reRaise, IExternalVariableList variableList);
		///<summary>
		/// these two overloads returns the <see cref="IStructuralFeature"/> (if any) that the result of evaluating the expression would have.
		///</summary>
		IStructuralFeature ExpressionModelInfo(string expression, IClassifier context, bool reRaise);
		IStructuralFeature ExpressionModelInfo(string expression, IClassifier context, bool reRaise, IExternalVariableList variableList);
		///<summary>
		/// these two overloads returns true if the result of evaluating the expression would be a readonly element.
		///</summary>
		bool ExpressionIsReadOnly(string expression, IClassifier context, bool reRaise);
		bool ExpressionIsReadOnly(string expression, IClassifier context, bool reRaise, IExternalVariableList variableList);
		///<summary>
		/// Returns the set of all installed operations in the ocl-evaluator. This method is mainly used by the Ocl editor.
		///</summary>
		IOclOperation[] InstalledOclOperations { get; }
		///<summary>
		/// Installs a new ocl <paramref name="operation"/> in the evaluator.
		///</summary>
		void InstallOperation(IOclOperation operation);
	}
	///<summary>
	/// The methods of this interface performs static analysis of ocl-expressions.
	/// Mainly intended to be used in designtime to calculate static types of handles
	///</summary>
	public interface IOclTypeService: ITypeService
	{
	}

	///<summary>
	/// This interface allows programmatic access to evaluating ocl expressions.
	///</summary>
	public interface IOclService: IOclTypeService
	{
		///<summary>
		/// Evaluates an expression in the context of the EcoSpace.
		///</summary>
		IElement Evaluate(string expression);
		///<summary>
		/// Evaluates an expression in the context of the EcoSpace with external variables.
		///</summary>
		IElement Evaluate(string expression, IExternalVariableList variableList);
		///<summary>
		/// Evaluates an expression in a specified context
		///</summary>
		IElement Evaluate(IElement root, string expression);
		///<summary>
		/// Evaluates an expression in a specified context with external variables
		///</summary>
		IElement Evaluate(IElement root, string expression, IExternalVariableList variableList);
		///<summary>
		/// Evaluates an expression with a specified root context and subscribers.
		///</summary>
		IElement EvaluateAndSubscribe(IElement root, string expression, ISubscriber reevaluateSubscriber, ISubscriber resubscribeSubscriber);
		///<summary>
		/// Evaluates an expression with a specified root context, variableList and subscribers.
		///</summary>
		IElement EvaluateAndSubscribe(IElement root, IExternalVariableList variableList, string expression, ISubscriber reevaluateSubscriber, ISubscriber resubscribeSubscriber);
		///<summary>
		/// Evaluates an expression with a specified root context, root type and subscribers.
		///</summary>
		IElement EvaluateAndSubscribe(IElement root, IClassifier rootType, string expression, ISubscriber reevaluateSubscriber, ISubscriber resubscribeSubscriber);
		///<summary>
		/// Evaluates an expression with a specified root context, root type, variableList and subscribers.
		///</summary>
		IElement EvaluateAndSubscribe(IElement root, IClassifier rootType, IExternalVariableList variableList, string expression, ISubscriber reevaluateSubscriber, ISubscriber resubscribeSubscriber);
		///<summary>
		/// Returns an element that will maintain internal subscriptions and always be kept up to date when
		/// changes in the underlying objectspace affects the result of the expression.
		///</summary>
		IElement GetDerivedElement(IElement root, string expression);
	}
	///<summary>
	/// The methods of this interface performs static analysis of ocl-expressions.
	/// Mainly intended to be used in designtime to calculate static types of handles
	///</summary>
	public interface IOclPsTypeService: ITypeService
	{
	}
	///<summary>
	/// This interface allows evaluation OCL expressions directly in the persistent
	/// storage (database). The OCL expression will be translated to SQL and executed directly
	/// in the database. There set of OCL expressions that can be evaluated in the database
	/// ios a subset of all expressions, and the result must always be a list of objects.
	/// When executing the query, only the identities of the objects will be fetched, not the
	/// contents.
	///</summary>
	public interface IOclPsService: IOclPsTypeService
	{
		///<summary>
		/// This method returns an abstract condition that represents the ocl-expression and that can be passed as a parameter to IPersistenceService.GetAllWithCondition
		/// this is the programmatic equivalence of using the OclPSHandle.
		///</summary>
		AbstractCondition CreateRetrieveCondition(IElement root, IExternalVariableList variableList, string expression);
		///<summary>
		/// Exaluate the OCL expression expression in the database. The expression must evaluate
		/// to a list of objects. 
		///</summary>
		IObjectList Execute(string expression);
		///<summary>
		/// Exaluate the OCL expression expression in the database. The expression must evaluate
		/// to a list of objects. The offset first objects of the result will not be included in the list
		/// and at most maxAnswers will be included.
		///</summary>		
		IObjectList Execute(string expression, int maxAnswers, int offset);
		///<summary>
		/// Exaluate the OCL expression expression in the databaseles,
		/// using root as the context.
		///</summary>		
		IObjectList Execute(IElement root, string expression);
		///<summary>
		/// Exaluate the OCL expression expression in the databaseles,
		/// using root as the context, and using the variables in variableList. The expression must evaluate
		/// to a list of objects. The offset first objects of the result will not be included in the list
		/// and at most maxAnswers will be included.
		///</summary>		
		IObjectList Execute(IElement root, IExternalVariableList variableList, string expression, int maxAnswers, int offset);
	}

	///<summary>
	/// The methods of this interface performs static analysis of ocl-expressions.
	/// Mainly intended to be used in designtime to calculate static types of handles
	///</summary>
	public interface IActionLanguageTypeService: ITypeService
	{
	}

	///<summary>
	/// This interface allows programmatic access to evaluating expressions in the
	/// Eco Action Language. The Languaage is a superset of OCL.
	///</summary>	

	public interface IActionLanguageService: IActionLanguageTypeService
	{
		///<summary>
		/// Executes an action in the context of the EcoSpace.
		///</summary>
		void Execute(string expression);
		///<summary>
		/// Executes an action in the context of the EcoSpace with external variables.
		///</summary>
		void Execute(string expression, IExternalVariableList variableList);
		///<summary>
		/// Executes an action in a specified context
		///</summary>
		void Execute(IElement root, string expression);
		///<summary>
		/// Executes an action in a specified context with external variables
		///</summary>
		void Execute(IElement root, string expression, IExternalVariableList variableList);
	}
	///<summary>
	/// This interface is mainly intended for the internal use of the ElementHandles.
	///</summary>
	public interface IStaticContext
	{
		IEcoTypeSystem TypeSystem { get; }
		IOclTypeService OclTypeService { get; }
		IOclPsTypeService OclPsTypeService { get; }
		IActionLanguageTypeService ActionLanguageTypeService { get; }
		IExternalVariableList VariableList { get; }
		IClassifier StaticUmlType { get; }
	}

	///<summary>
	///Allows you to create undo/redo blocks and transactions.
	///</summary>
	public interface IUndoService
	{
		///<summary>
		///returns a generated name that is unique in the undo/redo mechanism. If the suggested name is unused, it will be used, otherwise a counter will be appended to it.
		///</summary>
		string GetUniqueBlockName(string suggestedName);
		///<summary>
		///returns true if it is possible to move the named block to the top of the undo stack
		///<exception cref="System.ArgumentException"> thrown if blockName does not represent an existing block.</exception>
		///</summary>
		bool CanUndoBlock(string blockName);
		///<summary>
		///returns true if it is possible to move the named block to the top of the redo stack
		///<exception cref="System.ArgumentException"> thrown if blockName does not represent an existing block.</exception>
		///</summary>
		bool CanRedoBlock(string blockName);
		///<summary>
		///Undoes the top most block on the undo stack
		///<exception cref="System.InvalidOperationException"> thrown if system is in a transaction</exception>
		///</summary>
		void UndoLatest();
		///<summary>
		///Redoes the top most block on the redo stack
		///<exception cref="System.InvalidOperationException"> thrown if system is in a transaction</exception>
		///</summary>
		void RedoLatest();
		///<summary>
		///Undoes the block with the specified name (if it is possible to move it to the top of the undo stack first)
		///<exception cref="System.InvalidOperationException"> thrown if system is in a transaction.</exception>
		///<exception cref="System.ArgumentException"> thrown if blockName does not represent an existing block.</exception>
		///<exception cref="System.InvalidOperationException"> thrown if the block cannot be undone</exception>
		///</summary>
		void UndoBlock(string blockName);
		///<summary>
		///Redoes the block with the specified name (if it is possible to move it to the top of the redo stack first).
		///<exception cref="System.InvalidOperationException"> thrown if system is in a transaction</exception>
		///<exception cref="System.ArgumentException"> thrown if blockName does not represent an existing block.</exception>
		///<exception cref="System.InvalidOperationException"> thrown if the block cannot be redone</exception>
		///</summary>
		void RedoBlock(string blockName);
		///<summary>
		///Creates a new undo block with a specified name.
		///<exception cref="System.InvalidOperationException"> thrown if name is already in use.</exception>
		///<exception cref="System.ArgumentException"> thrown if blockName exists.</exception>
		///</summary>
		void StartUndoBlock(string blockName);
		///<summary>
		///Creates a new undo block with a generated (unique) name.
		///</summary>
		void StartUndoBlock();
		///<summary>
		///Removes all undoblocks from the undo stack.
		///</summary>
		void ClearAllUndoBlocks();
		///<summary>
		///Returns the current stack of undo blocks
		///</summary>
		IUndoBlockList UndoList { get; }
		///<summary>
		///Returns the current stack of redo blocks
		///</summary>
		IUndoBlockList RedoList { get; }
		///<summary>
		/// Starts in in-memory transaction. Must be matched by either a call to
		/// CommitTransaction or RollbackTransaction.
		/// Each call to StartTransaction will increment TransactionNesting.
		///</summary>
		void StartTransaction();
		///<summary>
		/// Commits the changes done since the matching call to startTransaction
		/// (unless a nested transaction has been rolled back)
		///</summary>
		void CommitTransaction();
		///<summary>
		/// Rolls back the changes done since the matching call to startTransaction
		/// (unless this is a nested transaction, then the changes will be rolled back when
		/// the first transaction is committed/rolledback)
		///</summary>
		void RollbackTransaction();     
		///<summary>
		/// True if serivice is in a transaction
		///</summary>
		bool IsInTransaction { get; }
		///<summary>
		/// Number of unmatched calls to StartTransaction
		///</summary>
		int TransactionNesting { get; }
	}

	///<summary>
	///Encapsulated a list/stack of undo blocks and operations on them.
	///</summary>
	public interface IUndoBlockList: IEnumerable
	{
		///<summary>
		///Renames an undo-block.
		///<param name="oldName">Original name of block.</param>
		///<param name="newName">New name of block.</param>
		///<exception cref="System.ArgumentException">thrown if there is a block named newName.</exception>
		///<exception cref="System.ArgumentException">thrown if there is no block named oldName.</exception>
		///</summary>
		void RenameBlock(string oldName, string newName);
		///<summary>
		///Removes the block named blockName
		///<param name="blockName">The name of the block to remove</param>
		///<returns>Returns true if the block could be deleted (requires that it can be moved to top)</returns>
		///<exception cref="System.ArgumentException">Thrown if blockName is not found</exception>
		///</summary>
		bool RemoveBlock(string blockName);
		///<summary>
		///Moves the block named blockName to top of the stack.
		///Implementation uses MoveBlock to accomplish its task.
		///<param name='blockName'>The name of the block to move</param>
		///</summary>
		void MoveToTop(string blockName);
		///<summary>
		///Returns the index of the block named blockName.
		///<param name="blockName">The name of the block for which to find the index</param>
		///<returns>Returns -1 if the blockName is not in the list</returns>
		///</summary>
		int IndexOf(string blockName);
		///<summary>
		///Examines if the block at curIndex can be merged with the block at newIndex
		///<param name="curIndex">The index of the block that should be tested</param>
		///<param name="newIndex">The index of the target position</param>
		///<returns>True if the merge is allowed</returns>
		///</summary>
		bool CanMergeBlock(int curIndex, int newIndex);
		///<summary>
		///Examines if the block at curIndex can be moved to newIndex
		///<param name="curIndex">The index of the block that should be tested</param>
		///<param name="newIndex">The index of the target position</param>
		///<returns>True if the move is allowed</returns>
		///</summary>
		bool CanMoveBlock(int curIndex, int newIndex);		
		///<summary>
		///Merges one undoblock into another.
		///<param name="destinationBlockName">The block name into which to merge</param>
		///<param name="sourceBlockName">The block name to merge.</param>
		///<exception cref="System.ArgumentException">thrown if destinationBlockName or sourceBlockName does not exist</exception>
		///<exception cref="System.InvalidOperationException">thrown if the blocks cannot be merged.</exception>
		///</summary>
		void MergeBlocks(string destinationBlockName, string sourceBlockName);
		///<summary>
		///Moves a block within the stack.
		///<param name="curIndex">The index of the block that should be moved</param>
		///<param name="newIndex">The index of the target position</param>
		///</summary>
		void MoveBlock(int curIndex, int newIndex);
		///<summary>
		///Number of blocks in list.
		///</summary>
		int Count { get; }
		///<summary>
		///Returns the block at position index.
		///<returns>Returns null if index is out of range.</returns>
		///</summary>
		IUndoBlock this[int index] { get; }
		///<summary>
		///Returns the block with name name.
		///<returns>Returns null if there is no block named name.</returns>
		///</summary>
		IUndoBlock this[string name] { get; }
		///<summary>
		///Returns the topmost undo-block. If there are no blocks, a new unnamed block will be created and returned.
		///</summary>
		IUndoBlock TopBlock { get; }
	}

	///<summary>
	///Encapsulates an undo-block. Undo-blocks are used by IUndoService.
	///</summary>
	public interface IUndoBlock
	{
		///<summary>
		///Name of block
		///</summary>
		string Name { get; }
		///<summary>
		///Returns true if the block contains any changes that can be undone.
		///</summary>
		bool ContainsChanges { get; }
		///<summary>
		///Returns the list of objects that are modified in this undo block.
		///</summary>
		IObjectList GetChangedObjects();
	}

	///<summary>
	/// <para>This service provides ids for objects that are guaranteed to be
	/// stable within the ecospace regardless of whether the objects have been stored or not.</para>
	/// <para>Normally, an object has an internal id when it is first created, and then
	/// changes id to a persistent id the first time it is stored in the database.</para>
	/// <para>This service is intended to be used by asp-clients so that the id handed to the web page
	/// can be converted back to the same object when an update is posted from the web page.</para>
	/// </summary>
	public interface IExternalIdService
	{
		///<summary>
		/// Creates a stable id for an object. The id is stable within the lifetime of the ecospace.
		///</summary>
		string IdForObject(IObject obj);
		///<summary>
		/// This method is guaranteed to return the same object that was used to create the id.
		///</summary>
		IObjectInstance ObjectForId(string id);
	}
	///<summary>
	/// This class contains a collection of useful utility methods for working with
	/// the services. All the methods are static.
	///</summary>	
	public sealed class EcoServiceHelper
	{
		private EcoServiceHelper() { }
		///<summary>
		/// Check if obj is of a type that can be used to reach a IEcoServiceProvider
		/// and if so return it.
		/// This method will be sucessfull if obj implements
		/// IObject, IProperty, IObjectProvder or IEcoServiceProvider.
		/// This includes Eco Objects, EcoSpaces and ElementHanldes.
		/// If obj is null an ArgumentNullException is raised.
		/// If obj cannot be used to reach a IEcoServiceProvider an ArgumentException is raised.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="obj"/> is null.</exception>
		///<exception cref="ArgumentException">Thrown if <paramref name="obj"/> obj cannot be used to reach a IEcoServiceProvider.</exception>
		public static IEcoServiceProvider GetServiceProvider(object obj)
		{
			if (obj == null)
				throw new ArgumentNullException("obj"); // do not localize
			if (obj is IObjectProvider)
				return (obj as IObjectProvider).AsIObject().ServiceProvider;
			else if (obj is IObject)
				return (obj as IObject).ServiceProvider;
			else if (obj is IProperty) 
				return GetServiceProvider((obj as IProperty).Owner);
			else if (obj is IEcoServiceProvider)
				return obj as IEcoServiceProvider;
			else
				throw new ArgumentException(
					InterfacesStringRes.sCantGetEcoSpaceProvider(obj.GetType().ToString()),
					"obj"); // do not localize
		}
		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IActionLanguageService from it.
		///</summary>
		public static IActionLanguageService GetActionLanguageService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IActionLanguageService)) as IActionLanguageService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IActionLanguageTypeService from it.
		///</summary>
		public static IActionLanguageTypeService GetActionLanguageTypeService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IActionLanguageTypeService)) as IActionLanguageTypeService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IDirtyListService from it.
		///</summary>
		public static IDirtyListService GetDirtyListService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IDirtyListService)) as IDirtyListService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IExtentService from it.
		///</summary>
		public static IExtentService GetExtentService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IExtentService)) as IExtentService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IExternalIdService from it.
		///</summary>
		public static IExternalIdService GetExternalIdService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IExternalIdService)) as IExternalIdService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IObjectFactoryService from it.
		///</summary>
		public static IObjectFactoryService GetObjectFactoryService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IObjectFactoryService)) as IObjectFactoryService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IOclPsService from it.
		///</summary>
		public static IOclPsService GetOclPsService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IOclPsService)) as IOclPsService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IOclPsTypeService from it.
		///</summary>
		public static IOclPsTypeService GetOclPsTypeService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IOclPsTypeService)) as IOclPsTypeService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IOclService from it.
		///</summary>
		public static IOclService GetOclService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IOclService)) as IOclService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IOclTypeService from it.
		///</summary>
		public static IOclTypeService GetOclTypeService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IOclTypeService)) as IOclTypeService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IPersistenceService from it.
		///</summary>
		public static IPersistenceService GetPersistenceService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IPersistenceService)) as IPersistenceService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IStateService from it.
		///</summary>
		public static IStateService GetStateService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IStateService)) as IStateService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// ITypeService from it.
		///</summary>
		public static ITypeService GetTypeService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(ITypeService)) as ITypeService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// ITypeSystemService from it.
		///</summary>
		public static ITypeSystemService GetTypeSystemService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(ITypeSystemService)) as ITypeSystemService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IUndoService from it.
		///</summary>
		public static IUndoService GetUndoService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IUndoService)) as IUndoService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IVariableFactoryService from it.
		///</summary>
		public static IVariableFactoryService GetVariableFactoryService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IVariableFactoryService)) as IVariableFactoryService;
		}

		///<summary>
		/// Get an IEcoServiceProvider using GetServiceProvider, and retrieve
		/// IVersionService from it.
		///</summary>
		public static IVersionService GetVersionService(object obj)
		{
			return GetServiceProvider(obj).GetEcoService(typeof(IVersionService)) as IVersionService;
		}
	}
}
